/************************************************************************\
**  ftp.c - File Transfert Protocol                                     **
**                                                                      **
**  Copyright (c) 2004 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/

	/* See RFC 959 for information on FTP
	   (with extensions from RFC 775)
	 */

	#include <ctype.h>
	#include <dirent.h>
	#include <errno.h>
	#include <getopt.h>
	#include <signal.h>
	#include <stdarg.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#include <time.h>
	#include <unistd.h>
	#include <arpa/inet.h>
	#include <netinet/in.h>
	#include <sys/socket.h>
	#include <sys/types.h>
	#include <sys/stat.h>
	#include <sys/fcntl.h>

	#include "multiserv.h"
	#include "multiserv_utils.h"

	#include "ftp.h"

#if defined(MULTISERV_FTP_TCP)

	static int	ftp_dialog (int sock, struct sockaddr_in * unused_client_address);


	static char *	ftp_getcwd(void);

	static int	ftp_check_access_setuid_chdir	(char * login, char * password);

	static int	ftp_quit  (int control_socket);
	static int	ftp_user  (int control_socket, char ** logname, char * arg);
	static int	ftp_pass  (int control_socket, char * logname, char * password);
	static int 	ftp_noop  (int control_socket);
	static int	ftp_syst  (int control_socket);
	static int	ftp_pwd   (int control_socket);
	static int	ftp_cwd   (int control_socket, char * pathname);
	static int	ftp_cdup  (int control_socket);
	static int	ftp_mkd   (int control_socket, char * directory);
	static int	ftp_rmd   (int control_socket, char * directory);
	static int	ftp_dele  (int control_socket, char * file);
	static int	ftp_list  (int control_socket, int * data_socket, char * arg, int long_format);
	static int	ftp_pasv  (int control_socket, int * data_socket);
	static int	ftp_port  (int control_socket, int * data_socket, char * string);
	
	static int	ftp_check_data_socket (int control_socket, int * data_socket, int passive);
	static char *	ftp_perm  (int perm);

	int
ftp_description (FILE * stream)
{
	return (fprintf(stream, "File Transfert Protocol (FTP) rfc.959 (TCP/IP)"));
}


	int
ftp_main (int argc, char * argv[])
{
	multiserv_log(MULTISERV_LOG_MESSAGE, "ftp", "TCP server start");
	return (tcp_server(ftp_dialog));
}


	static int
ftp_send_string(int sock, char * string)
{
	int n = strlen(string);
	if (send(sock, string, n, 0) == n)
		return (0);
	return (-1);
}

	static int
ftp_wait_string (int sock)
{
	int n;
	do {
		n = recv(sock, multiserv_buffer,MULTISERV_BUFFER_LEN, 0);
		if (n <= 0)
			return (-1);
		do {
			multiserv_buffer[n] = '\0';
			n --;
		} while ((n>=0) && (isspace(multiserv_buffer[n])));
	} while (n < 0);
	return (0);
}

	static int
ftp_dialog (int control_socket, struct sockaddr_in * client_address)
{
	char *	arg = NULL;
	char *	logname = NULL;
	int	access_granted = 0;
	struct	sockaddr_in	destination;
	int	data_socket = -1;
	int	passive = 0;
	
	multiserv_debug("ftp", "Got a connection, sending banner");	
	#ifdef MULTISERV_FTP_VERBOSE
		ftp_send_string(control_socket, "220 Multiserv FTP server ready\r\n");
	#else
		ftp_send_string(control_socket, "220 \r\n");
	#endif	
	memset(&destination, 0, sizeof(struct sockaddr_in));
	
	while(1) {
	
		/* ## TODO : add a timeout */
		if (ftp_wait_string(control_socket) != 0)
			break;

		arg = multiserv_buffer;
		while ((arg[0] != '\0') && (arg[0] != ' '))
			arg ++;
		while(arg[0] == ' ')
			arg ++;
fprintf(stderr, "[%s]\n", multiserv_buffer);
		/* QUIT is always available */
		if (strncasecmp (multiserv_buffer, "QUIT", 4) == 0) {
			ftp_quit(control_socket);
			break;
		}

		/* USER / PASS is only allowed when access is NOT already granted,
		   because we chroot during some connections */
		if (! access_granted) {
			if (strncasecmp (multiserv_buffer, "USER ", 5) == 0) {
				if (ftp_user(control_socket, & logname, arg) != 0)
					break;
				continue;
			}

			if (strncasecmp(multiserv_buffer, "PASS ", 5) == 0) {
				access_granted = (ftp_pass(control_socket, logname, arg) == 0);
				memset(multiserv_buffer, 0, MULTISERV_BUFFER_LEN); /* paranoia */
				continue;
			}

			/* All other commands need preliminary login */
			#ifdef MULTISERV_FTP_VERBOSE
				ftp_send_string(control_socket, "530 You need to log in first!\r\n");
			#else
				ftp_send_string(control_socket, "530\r\n");
			#endif
			continue;
		}
		
		if ((strcasecmp(multiserv_buffer, "NOOP") == 0) 
		||  (strncasecmp(multiserv_buffer, "ALLO", 4) == 0)) {
			if (ftp_noop(control_socket) != 0)
				break;
			continue;
		}
				
		if (strcasecmp(multiserv_buffer, "SYST") == 0) {
			if (ftp_syst(control_socket) != 0)
				break;
			continue;
		}
				
		if ((strcasecmp(multiserv_buffer, "PWD") == 0)
		 || (strcasecmp(multiserv_buffer, "XPWD") == 0)) {
			if (ftp_pwd(control_socket) != 0)
				break;
			continue;
		}

		if ((strncasecmp(multiserv_buffer, "CWD ", 4) == 0)
		 || (strncasecmp(multiserv_buffer, "XCWD ", 5) == 0)) {
		 	if (ftp_cwd(control_socket, arg) != 0)
				break;
			continue;
		}

		if ((strcasecmp(multiserv_buffer, "CDUP") == 0)
		|| (strcasecmp(multiserv_buffer, "XCUP") == 0)) {
			if (ftp_cdup(control_socket) != 0)
				break;
			continue;
		}

		if ((strncasecmp(multiserv_buffer, "MKD ", 4) == 0)
		 || (strncasecmp(multiserv_buffer, "XMKD ", 5) == 0)) {
		 	if (ftp_mkd(control_socket, arg) != 0)
				break;
			continue;
		}


		if ((strncasecmp(multiserv_buffer, "RMD ", 4) == 0)
		 || (strncasecmp(multiserv_buffer, "XRMD ", 5) == 0)) {
		 	if (ftp_rmd(control_socket, arg) != 0)
				break;
			continue;
		}
		
		if (strncasecmp(multiserv_buffer, "DELE ", 5) == 0) {
			if (ftp_dele(control_socket,arg) != 0)
				break;
			continue;
		}

		if (strncasecmp(multiserv_buffer, "PORT ", 5) == 0) {
			passive = 0;
			if (ftp_port(control_socket, & data_socket, arg) != 0)
				break;
			continue;
		}

		if (strcasecmp(multiserv_buffer, "PASV") == 0) {
			passive = 1;
			if (ftp_pasv(control_socket, & data_socket) != 0)
				break;
			continue;
		}
		

		if (strncasecmp(multiserv_buffer,"LIST", 4) == 0) {
			if (ftp_check_data_socket(control_socket, & data_socket, passive) != 0)
				continue;
			if (ftp_list(control_socket, & data_socket, arg, 1) != 0)
				break;
			continue;
		}
		if (strncasecmp(multiserv_buffer,"NLST", 4) == 0) {
			if (ftp_check_data_socket(control_socket, & data_socket, passive) != 0)
				continue;
			if (ftp_list(control_socket, & data_socket, arg, 0) != 0)
				break;
			continue;
		}

		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "500 Command unknown\r\n");
		#else
			ftp_send_string(control_socket, "500 \r\n");
		#endif
fprintf(stderr,"UNKNOWN : %s\n", multiserv_buffer);		
	}
	if (logname != NULL)
		free(logname);
	logname = NULL;
	return (0);
}

	static int
ftp_quit(int control_socket)
{
	#ifdef MULTISERV_FTP_VERBOSE
		return ftp_send_string(control_socket, "221 Bye.\r\n");
	#else
		return ftp_send_string(control_socket, "221 \r\n");
	#endif
}
		
	static int
ftp_user(int control_socket, char ** logname, char * arg)
{
	if (logname == NULL) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error (USER1)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	if (*logname != NULL)
		free (* logname);
	* logname = malloc (strlen(arg) + 1);
	if (* logname == NULL) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error (USER2)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	strcpy(* logname, arg);
	#ifdef MULTISERV_FTP_ANONYMOUS
		if (strcmp(* logname, "anonymous") == 0)
			#ifdef MULTISERV_FTP_VERBOSE
				ftp_send_string(control_socket, "331 Please send email adress as password\r\n");
			#else
				ftp_send_string(control_socket, "331 \r\n");
			#endif
	#endif
	#ifdef MULTISERV_FTP_VERBOSE
		ftp_send_string(control_socket, "331 User name ok, need password.\r\n");
	#else
		ftp_send_string(control_socket, "331 \r\n");
	#endif
	return (0);
}

	static int
ftp_pass(int control_socket, char * logname, char * password)
{
	if (logname == NULL) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "503 Please log in with USER first\r\n");
		#else
			ftp_send_string(control_socket, "503 \r\n");
		#endif
		return (-1);
	}
	#ifdef MULTISERV_FTP_ANONYMOUS
		if (strcmp(logname, "anonymous") == 0) {
			/* Anonymous ftp is only allowed for server ran by root */
			if ((chdir(MULTISERV_FTP_ANONYMOUS_DIR) != 0)
			 || (chroot(MULTISERV_FTP_ANONYMOUS_DIR) != 0)
			 || (setgid(MULTISERV_FTP_ANONYMOUS_GID) != 0)
			 || (setuid(MULTISERV_FTP_ANONYMOUS_UID) != 0)) {
				#ifdef MULTISERV_FTP_VERBOSE
					ftp_send_string(control_socket, "421 Service not available\r\n");
				#else
					ftp_send_string(control_socket, "421 \r\n");
				#endif
				return (-1);
			}
			#ifdef MULTISERV_FTP_VERBOSE
				ftp_send_string(control_socket, "230 Welcome A.N.Onymous.\r\n");
			#else
				ftp_send_string(control_socket, "230 \r\n");
			#endif
			/* ## log connection */
			return (0);
		}
	#endif
	if (ftp_check_access_setuid_chdir(logname, password)) {
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "230 User logged in, proceed\r\n");
		#else
			return ftp_send_string(control_socket, "230 \r\n");
		#endif
	}
	#ifdef MULTISERV_FTP_VERBOSE
		ftp_send_string(control_socket, "530 Login incorrect\r\n");
	#else
		ftp_send_string(control_socket, "530 \r\n");
	#endif
	return (1);
}
			
	static int
ftp_check_access_setuid_chdir(char * login, char * password)
{
	/* hmmm, so, what we gonna do ? */
	
	/* lazy... */
	return (1);
}

	static char *
ftp_getcwd(void)
{
	char * pathname = NULL;
	char * new;
	int	size = 0;
	
	do {
		size += 4096;
		new = realloc (pathname, size);
		if (new == NULL) {
			if (pathname != NULL)
				free(pathname);
			return (NULL);
		} 
		pathname = new;
	} while (getcwd(pathname, size) == NULL);
	return (pathname);
}


	static int 
ftp_noop(int control_socket)
{
	#ifdef MULTISERV_FTP_VERBOSE
		return ftp_send_string(control_socket, "200 Ok\n\n");
	#else
		return ftp_send_string(control_socket, "200 \n\n");
	#endif
}
	
	static int
ftp_syst(int control_socket)
{
	return ftp_send_string(control_socket, "215 LINUX\r\n");
}

	static int
ftp_pwd(int control_socket)
{
	char * pathname = ftp_getcwd();
	if (pathname == NULL) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	if (snprintf(multiserv_buffer, MULTISERV_BUFFER_LEN, "257 \"%s\"\r\n", pathname) >= MULTISERV_BUFFER_LEN) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		free(pathname);
		pathname = NULL;
		return (-1);
	}
	free(pathname);
	pathname = NULL;
	return ftp_send_string(control_socket, multiserv_buffer);
}

	static int
ftp_cwd(int control_socket, char * pathname)
{
	if (chdir(pathname) != 0)
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "550 Unable to change directory\r\n");
		#else
			return ftp_send_string(control_socket, "550 \r\n");
		#endif
	else
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "250 Directory Ok\r\n");
		#else
			return ftp_send_string(control_socket, "250 \r\n");
		#endif
}

	static int
ftp_cdup(int control_socket)
{
	if (chdir("..") != 0)
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "550 Unable to change directory\r\n");
		#else
			return ftp_send_string(control_socket, "550 \r\n");
		#endif
	else
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "250 Directory Ok\r\n");
		#else
			return ftp_send_string(control_socket, "250 \r\n");
		#endif
}



	static int
ftp_mkd(int control_socket, char * directory)
{
	int fd;
	char * pathname;
	if (mkdir(directory, 0755) != 0) {
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "550 Creation failed\r\n");
		#else
			return ftp_send_string(control_socket, "550 d\r\n");
		#endif
	}
	if ((fd = open (".", O_RDONLY)) < 0) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	if ((chdir(directory) != 0)
	 || ((pathname = ftp_getcwd()) == NULL)) {
	 	fchdir(fd);
	 	close(fd);
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	if (snprintf(multiserv_buffer, MULTISERV_BUFFER_LEN, "257 \"%s\"\r\n", pathname) >= MULTISERV_BUFFER_LEN) { 
		fchdir(fd);
		close(fd);
		free(pathname);
		pathname = NULL;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	fchdir(fd);
	close(fd);
	free(pathname);
	pathname = NULL;
	return ftp_send_string(control_socket, multiserv_buffer);
}
	static int
ftp_rmd(int control_socket, char * directory)
{
	if (rmdir(directory) != 0)
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "550 Delete failed\r\n");
		#else
			return ftp_send_string(control_socket, "550 \r\n");
		#endif
	else
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "250 Delete ok\r\n");
		#else
			return ftp_send_string(control_socket, "250 \r\n");
		#endif
}

	static int
ftp_dele(int control_socket, char * file)
{
	if (unlink(file) != 0)
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "550 Delete failed\r\n");
		#else
			return ftp_send_string(control_socket, "550 \r\n");
		#endif
	else
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "250 Delete ok\r\n");
		#else
			return ftp_send_string(control_socket, "250 \r\n");
		#endif
}


	static int
ftp_list(int control_socket, int * data_socket, char * arg, int long_format)
{
	char * directory = ".";
	DIR * dp;
	struct dirent * entry;
	struct stat filestat;
	struct tm * tm;
	int option_l = 0;
	int option_a = 0;

	if (arg[0] == '-') {
		if (strchr(arg, 'l') != NULL)
			option_l = 1;
		if (strchr(arg, 'a') != NULL)
			option_a = 1;
	} else {		
		if (arg[0] != '\0')
			directory = arg;
	}
	dp = opendir(directory);
	if (dp == NULL) {
		#ifdef MULTISERV_FTP_VERBOSE
			return ftp_send_string(control_socket, "550 No such file or directory\r\n");
		#else
			return ftp_send_string(control_socket, "550 \r\n");
		#endif
	}
	#ifdef MULTISERV_FTP_VERBOSE
		ftp_send_string(control_socket, "150 Here comes the directory listing\r\n");
	#else
		ftp_send_string(control_socket, "150 \r\n");
	#endif
	while ((entry = readdir(dp)) != NULL) {
		if ((entry->d_name[0] == '.') && (! option_a ))
		 	continue;
		if ((! long_format) && (! option_l)) {
			if (snprintf(multiserv_buffer, MULTISERV_BUFFER_LEN, "%s\r\n", entry->d_name) >= MULTISERV_BUFFER_LEN)
				continue;
		} else {
			if (snprintf(multiserv_buffer, MULTISERV_BUFFER_LEN, "%s/%s", directory, entry->d_name) >= MULTISERV_BUFFER_LEN)
				continue;
			if (lstat(multiserv_buffer, & filestat) < 0)
				continue;
			tm = localtime(& filestat.st_mtime);
			if (snprintf(multiserv_buffer,
					MULTISERV_BUFFER_LEN,
					"%s %4d %4d %4d %9ld %04d/%02d/%02d %02d:%02d:%02d %s\r\n",
					ftp_perm(filestat.st_mode) ,
					filestat.st_nlink, filestat.st_uid, filestat.st_gid, filestat.st_size,
					tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
					tm->tm_hour, tm->tm_min, tm->tm_sec,
					entry->d_name) >= MULTISERV_BUFFER_LEN)
				continue;
		}
		if (write(* data_socket, multiserv_buffer, strlen(multiserv_buffer)) != strlen(multiserv_buffer)) {
			break;
		}
	}
	close(* data_socket);
	* data_socket = -1;
	closedir(dp);
	if (entry != NULL) { /* error in ftp_send */
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (1);
	}
	#ifdef MULTISERV_FTP_VERBOSE
		return ftp_send_string(control_socket, "226 Data transfert complete\r\n");
	#else
		return ftp_send_string(control_socket, "226 \r\n");
	#endif
}



	static int
ftp_pasv(int control_socket, int * data_socket)
{
	/* 
	 * This routine initialize a passive data socket, using the same IP
	 * address as the control socket and a random port (provided by the
	 * kernel).
	 * The data_socket argument is filled in return with the new socket
	 * that has to be accept()ed before real data transfert.
	 */
	struct sockaddr_in address;
	socklen_t namelen;
	
	if (* data_socket != -1)
		close(* data_socket);
	* data_socket = -1;
	
	* data_socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (* data_socket < 0) {
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error (PASV1)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	namelen = sizeof(struct sockaddr_in);
	if (getsockname(control_socket, (struct sockaddr *) & address, & namelen) < 0) {
		close (* data_socket);
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 local error (PASV2)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	address.sin_port = 0;
	if (bind(* data_socket, (struct sockaddr *) & address, sizeof(struct sockaddr_in)) < 0) {
		close (* data_socket);
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 local error (PASV3)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	if (listen(* data_socket, 5) != 0) {
		close (* data_socket);
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 Local error (PASV4)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	namelen = sizeof(struct sockaddr_in);
	if (getsockname(* data_socket, (struct sockaddr *) & address, & namelen) < 0) {
		close (* data_socket);
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 local error (PASV5)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	if (snprintf(multiserv_buffer, MULTISERV_BUFFER_LEN, "227 Entering Passive Mode (%d,%d,%d,%d,%d,%d)\r\n",
		(ntohl(address.sin_addr.s_addr) >> 24) & 0xFF,
		(ntohl(address.sin_addr.s_addr) >> 16) & 0xFF,
		(ntohl(address.sin_addr.s_addr) >> 8) & 0xFF,
		(ntohl(address.sin_addr.s_addr)) & 0xFF,
		(ntohs(address.sin_port) >> 8) & 0xFF,
		(ntohs(address.sin_port)) & 0xFF) >= MULTISERV_BUFFER_LEN) {
		close (* data_socket);
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 local error (PASV6)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	if (ftp_send_string(control_socket, multiserv_buffer)  != 0) {
		close (* data_socket);
		* data_socket = -1;
		return (-1);
	}
	return (0);
}

	static int
ftp_port(int control_socket, int * data_socket, char * string)
{
	struct sockaddr_in address;
	int a1, a2, a3, a4;
	int p1, p2;

	if (* data_socket != -1)
		close (* data_socket);
	* data_socket = -1;
	if ((sscanf(string, "%d,%d,%d,%d,%d,%d", & a1, & a2, & a3, & a4, & p1, & p2) != 6)
	 || (a1<0) || (a2<0) || (a3<0) || (a4<0) || (a1>255) || (a2>255) || (a3>255) || (a4>255)) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "500 Illegal PORT command\r\n");
		#else
			ftp_send_string(control_socket, "500 \r\n");
		#endif
		return (0);
	}

	* data_socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (* data_socket < 0) {
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "451 local error (PORT1)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
		return (-1);
	}
	memset(& address, 0, sizeof(struct sockaddr_in));
	address.sin_family = AF_INET;
	address.sin_port = htons(p1*256+p2);
	address.sin_addr.s_addr = htonl(a1*16777216+a2*65536+a3*256+a4);

	if (connect(* data_socket, (struct sockaddr *) & address, sizeof(struct sockaddr_in)) < 0) {	
		close (* data_socket);
		* data_socket = -1;
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "425 local error (PORT2)\r\n");
		#else
			ftp_send_string(control_socket, "451 \r\n");
		#endif
	}
	#ifdef MULTISERV_FTP_VERBOSE
		ftp_send_string(control_socket, "200 PORT command Ok\r\n");
	#else
		ftp_send_string(control_socket, "200 \r\n");
	#endif
	return(0);
}


	static int
ftp_check_data_socket(int control_socket, int * data_socket, int passive)
{
	int	new_socket;
	if (* data_socket < 0) {
		#ifdef MULTISERV_FTP_VERBOSE
			ftp_send_string(control_socket, "425 Use PORT or PASV first\r\n");
		#else
			ftp_send_string(control_socket, "425 \r\n");
		#endif
		return (-1);
	}
	if (! passive)
		return (0);
	if ((new_socket = accept(* data_socket, NULL, NULL)) < 0) {
		close(* data_socket);
		* data_socket = -1;
		return (-1);
	}
	close(* data_socket);
	* data_socket = new_socket;
	return(0);
}

	static char *
ftp_perm(int perm)
{
	static char ret[11];

	if (S_ISREG(perm))       ret[0] = '-';
	else if (S_ISDIR(perm))  ret[0] = 'd';
	else if (S_ISCHR(perm))  ret[0] = 'c';
	else if (S_ISBLK(perm))  ret[0] = 'b';
	else if (S_ISFIFO(perm)) ret[0] = 'p';
	else if (S_ISLNK(perm))  ret[0] = 'l';
	else if (S_ISSOCK(perm)) ret[0] = 's';
	else                     ret[0] = '?';

	if (perm & 0400) ret[1]='r'; else ret[1] = '-';
	if (perm & 0200) ret[2]='w'; else ret[2] = '-';
	if (perm & 0100) ret[3]='x'; else ret[3] = '-';
	if (perm & 0040) ret[4]='r'; else ret[4] = '-';
	if (perm & 0020) ret[5]='w'; else ret[5] = '-';
	if (perm & 0010) ret[6]='x'; else ret[6] = '-';
	if (perm & 0004) ret[7]='r'; else ret[7] = '-';
	if (perm & 0002) ret[8]='w'; else ret[8] = '-';
	if (perm & 0001) ret[9]='x'; else ret[9] = '-';
	ret[9] = '\0';
	return(ret);
}


#endif
