/************************************************************************\
**  multiserv.c - Multiple Internet Servers for Embedded Systems        **
**                                                                      **
**  Copyright (c) 2004 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/


	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#include <time.h>
	#include <unistd.h>

	#include "multiserv.h"

	char * multiserv_author = "Christophe Blaess";

	char * multiserv_hostname = NULL;
	char * multiserv_portname = NULL;
	int  multiserv_uid = -1;
	int  multiserv_gid = -1;

	#if defined(MULTISERV_CHARGEN_TCP) || defined(MULTISERV_CHARGEN_UDP)
	#	include "chargen.h"
	#endif
	#if defined(MULTISERV_DAYTIME_TCP) || defined(MULTISERV_DAYTIME_UDP)
	#	include "daytime.h"
	#endif
	#if defined(MULTISERV_DISCARD_TCP) || defined(MULTISERV_DISCARD_UDP)
	#	include "discard.h"
	#endif
	#if defined(MULTISERV_ECHO_TCP) || defined(MULTISERV_ECHO_UDP)
	#	include "echo.h"
	#endif
	#if defined(MULTISERV_FTP_TCP)
	#	include "ftp.h"
	#endif
	#if defined(MULTISERV_QOTD_TCP) || defined(MULTISERV_QOTD_UDP)
	#	include "qotd.h"
	#endif
	#if defined(MULTISERV_TELNET_TCP)
	#	include "telnet.h"
	#endif
	#if defined(MULTISERV_TFTP_TCP) || defined(MULTISERV_TFTP_UDP)
	#	include "tftp.h"
	#endif
	#if defined(MULTISERV_TIME_TCP) || defined(MULTISERV_TIME_UDP)
	#	include "time.h"
	#endif
	
	#ifdef MULTISERV_LOG
		static FILE * log_fp = NULL;
	#endif
	
	static int	multiserv_main	(int argc, char * argv[]);
	static void	multiserv_help	(char * name);

	static struct s_service {
		char *	name;
		char *	alias;
		char *	port;
		int		(* main) (int, char **);
		int		(* description)(FILE * output);
	} services[] = {

		{ "multiserv",		NULL, 			NULL,	multiserv_main,		NULL },

		#ifdef MULTISERV_CHARGEN_TCP
		{ "chargen-tcp",	"chargen",		"19",	chargen_tcp_main,	chargen_tcp_description	},
		#endif
		#ifdef MULTISERV_CHARGEN_UDP
		{ "chargen-udp",	NULL,			"19",	chargen_udp_main,	chargen_udp_description	},
		#endif
		#ifdef MULTISERV_DAYTIME_TCP
		{ "daytime-tcp",	"daytime",		"13",	daytime_tcp_main, 	daytime_tcp_description	},
		#endif
		#ifdef MULTISERV_DAYTIME_UDP
		{ "daytime-udp",	NULL, 			"13",	daytime_udp_main, 	daytime_udp_description},
		#endif
		#ifdef MULTISERV_DISCARD_TCP
		{ "discard-tcp",	"discard",		"9",	discard_tcp_main, 	discard_tcp_description	},
		#endif
		#ifdef MULTISERV_DISCARD_UDP
		{ "discard-udp",	NULL, 			"9",	discard_udp_main, 	discard_udp_description	},
		#endif
		#ifdef MULTISERV_ECHO_TCP
		{ "echo-tcp",		"echo", 		"7",	echo_tcp_main, 		echo_tcp_description	},
		#endif
		#ifdef MULTISERV_ECHO_UDP
		{ "echo-udp",		NULL, 			"7", 	echo_udp_main, 		echo_udp_description	},
		#endif
		#ifdef MULTISERV_FTP_TCP
		{ "ftp-tcp",		"ftp", 			"21",	ftp_main,	 		ftp_description			},
		#endif
		#ifdef MULTISERV_QOTD_TCP
		{ "qotd-tcp",		"qotd", 		"17", 	qotd_tcp_main, 		qotd_tcp_description	},
		#endif
		#ifdef MULTISERV_QOTD_UDP
		{ "qotd-udp",		NULL, 			"17", 	qotd_udp_main, 		qotd_udp_description	},
		#endif
		#ifdef MULTISERV_TELNET_TCP
		{ "telnet-tcp",		"telnet",		"23", 	telnet_main, 		telnet_description		},
		#endif
		#ifdef MULTISERV_TFTP_TCP
		{ "tftp-tcp",		NULL, 			"69",	tftp_tcp_main, 		tftp_tcp_description	},
		#endif
		#ifdef MULTISERV_TFTP_UDP
		{ "tftp-udp",		"tftp",			"69",	tftp_udp_main,		tftp_udp_description	},
		#endif
		#ifdef MULTISERV_TIME_TCP
		{ "time-tcp",		"time",			"37",	time_tcp_main, 		time_tcp_description	},
		#endif
		#ifdef MULTISERV_TIME_UDP
		{ "time-udp",		NULL,			"37",	time_udp_main, 		time_udp_description	},
		#endif

		{ "",			NULL}
	};

	int
main (int argc, char * argv[])
{
	char * server_name;
	struct s_service * serv;
	int i,j,k;	
	
	#ifdef MULTISERV_LOG
		if (log_fp == NULL) {
			log_fp = fopen (MULTISERV_LOG_FILE, "a");
			if (log_fp == NULL) {
				fprintf (stderr, "Unable to open log file %s\n", MULTISERV_LOG_FILE);
				exit (1);
			}
		}
	#endif

	if (! strcmp(argv[0], "-h")) {
		multiserv_usage_global_options("multiserv");
		return (0);
	}

	/* server_name = basename(argv[0]) */
	for (i = strlen(argv[0]) - 1; i >= 0; i --)
		if (argv[0][i] == '/')
			break;
	server_name = & (argv[0][i+1]);
	
	for (serv = services; serv->name[0] != '\0'; serv++)
		if ((strcmp(server_name, serv->name) == 0)
		 || ((serv->alias != NULL)
		  && (strcmp(server_name, serv->alias) == 0)))
			break;
	if (serv->name[0] == '\0') {
		multiserv_log(MULTISERV_LOG_ERROR, server_name, "server not found");
		return (1);
	}
	
	if (multiserv_portname == NULL)
		multiserv_portname = serv->port;
	/* options... */
	i = 1;
	j = 1;
	while (i < argc) {
		if (argv[i][0] != '-') {
			i ++;
			continue;
		}
		switch (argv[i][j]) {
			case 'p' :
			case 'a' :
			case 'u' :
			case 'g' :
				if ((argv[i][j+1] != '\0')
				 || (i == argc - 1)) {
				 	multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "Option needs an argument\n");
					exit (1);
				}
			default :
				break;
		}
		
		switch (argv[i][j]) {
			case 'p' :		
				multiserv_portname = argv[i+1];
				break;
			case 'a' :
				multiserv_hostname = argv[i+1];
				break;
			case 'u' : 
				if (sscanf (argv[i+1], "%d", & multiserv_uid) != 1) {
				 	multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "Wrong arg for option -u\n");
					exit (1);
				}
				break;
			case 'g' :
				if (sscanf (argv[i+1], "%d", & multiserv_gid) != 1) {
				 	multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "Wrong arg for option -g\n");
					exit (1);
				}
				break;
			default :
				j ++;
				if (argv[i][j] =='\0') {
					j = 1;
					i ++;
				}
				continue;
		}

		for (k = i + 2; k < argc; k ++)
			argv[k - 2] = argv[k];
		j = 1;
		argc -= 2;
	}
	return (serv->main(argc, argv));
}

	void
multiserv_usage_global_options(char * server)
{
	if (! isatty(STDERR_FILENO))
		return;
	fprintf (stderr, "usage: %s [options]\n", server);
	if (multiserv_portname != NULL)
		fprintf (stderr, "  -p <port>  Local port instead of %s\n", multiserv_portname);
	else
		fprintf (stderr, "  -p <port>  Local port for server\n");
	fprintf (stderr, "  -a <addr>  Local address instead of ANY\n");
	fprintf (stderr, "  -u <uid>   User ID for server process\n");
	fprintf (stderr, "  -g <gid>   Group ID for server process\n");
	fprintf (stderr, "  -h         Help screen\n");
}


#ifdef MULTISERV_LOG
	void
multiserv_log (int level, char * server, char * message)
{
	time_t now;
	struct tm tim;
	
	#ifdef NDEBUG
		if (level > MULTISERV_LOG_ERROR)
			return;
	#endif
	if (level == MULTISERV_LOG_ERROR)
		if (isatty(STDERR_FILENO))
			fprintf (stderr, "%s: %s\n", server, message);
	time (& now);
	localtime_r(&now, &tim);
	fprintf (log_fp, "%04d/%02d/%02d %02d:%02d:%02d [%s] %s\n",
					tim.tm_year+1900,
					tim.tm_mon+1,
					tim.tm_mday,
					tim.tm_hour,
					tim.tm_min,
					tim.tm_sec,
					server, message);
}
#else
	void
multiserv_log (int level, char * server, char * message)
{
	if (level == MULTISERV_LOG_ERROR)
		if (isatty(STDERR_FILENO))
			fprintf (stderr, "%s: %s\n", server, message);
}
#endif

	static int
multiserv_main (int argc, char * argv[])
{
	int i, j;
	if (argc == 1) {
		multiserv_help(argv[0]);
		return (0);
	}
	
	for (j = 0, i = 1; i < argc; j++, i ++) {
		argv[j] = argv[i];
	}
	argc = j;
	return (main(argc, argv));
}

	static void
multiserv_help (char * name)
{
	struct s_service * serv;
	if (! isatty(STDERR_FILENO))
		return;
	fprintf(stderr, "MultiServ v.%s - (c) 2003-2004 %s\n", MULTISERV_VERSION, multiserv_author);
	fprintf(stderr, "\nusage: %s server [server_options...]\n", name);
	fprintf(stderr, "\nknown servers:\n");

	for (serv = services; serv->name[0] != '\0'; serv ++) {
		if (serv->description != NULL) {
			fprintf(stderr, "  %-12s", serv->name);
			if (serv->alias != NULL)
				fprintf(stderr, " %-12s", serv->alias);
			else
				fprintf(stderr, "             ");
			serv->description(stderr);
			fprintf(stderr, "\n");
		}
	}
}

