/************************************************************************\
**  multiserv_utils.c                                                   **
**                                                                      **
**  Copyright (c) 2004 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/

	#include <errno.h>
	#include <netdb.h>
	#include <signal.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#include <unistd.h>
	#include <arpa/inet.h>
	#include <sys/socket.h>
	#include <sys/wait.h>

	#include "multiserv.h"
	#include "multiserv_utils.h"

	char multiserv_buffer [MULTISERV_BUFFER_LEN];


	static int		change_uid_gid				(void);
	static int		install_signal_handlers		(void);
	static 	int		give_ip_address				(char * hostname, struct in_addr * result);
	static	int		give_port_number			(char * portname, char * protocol);	
	static	void	no_zombie_sigcld_handler	(int unused);
	static	void	empty_sigalrm_handler		(int unused);
	static int		create_and_bind_socket 		(int socket_type);




	int
udp_server (int (*function)(int socket, struct sockaddr_in * client, int msg_size))
{
	struct sockaddr_in	client_address;
	int					sock;

	if (install_signal_handlers() != 0)
		return (1);

	if ((sock = create_and_bind_socket (MULTISERV_UDP))  < 0)
		return (1);

	if (change_uid_gid () != 0)
		return (1);
	
	/* Then wait for incoming messages, fork, and let the child process the dialog */
	while (1) {
		int msg_size;
		size_t size = sizeof (struct sockaddr_in);
		msg_size = recvfrom(sock, multiserv_buffer, MULTISERV_BUFFER_LEN, 0, (struct sockaddr *) & client_address, & size);
		if (msg_size < 0) {
			if (errno == EINTR) /* SIGCLD */
				continue;
			return (1);
		}
		switch (fork()) {
			case -1 :
				multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to create a new child process");
				return (1);
			case 0 :
				close(sock);
				sock = socket(AF_INET, SOCK_DGRAM, 0);
				if (sock < 0) {
					multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to get a new UDP/IP socket");
					exit (1);
				}
				if (connect(sock, (struct sockaddr *) & client_address, sizeof(struct sockaddr_in)) < 0) {
					multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to use UDP connection to the client");
					exit (1);
				}
				exit (function(sock, & client_address, msg_size));
			default :
				break;
		}
	}
}

	int
udp_simple_server (int (*function)(int socket, struct sockaddr_in * client, int msg_size))
{
	struct sockaddr_in	client_address;
	int					sock;

	if (install_signal_handlers() != 0)
		return (1);

	if ((sock = create_and_bind_socket (MULTISERV_UDP))  < 0)
		return (1);

	if (change_uid_gid () != 0)
		return (1);

	while (1) {
		int msg_size;
		size_t size = sizeof (struct sockaddr_in);
		msg_size = recvfrom(sock, multiserv_buffer, MULTISERV_BUFFER_LEN, 0, (struct sockaddr *) & client_address, & size);
		if (msg_size < 0) {
			if (errno == EINTR) /* SIGCLD ? */
				continue;
			return (1);
		}
		function(sock, & client_address, msg_size);
	}
}


	int
tcp_server (int (*function)(int socket, struct sockaddr_in * client))
{
	struct sockaddr_in	client_address;
	int					sock;

	if (install_signal_handlers() != 0)
		return (1);

	if ((sock = create_and_bind_socket (MULTISERV_TCP))  < 0)
		return (1);

	if (change_uid_gid () != 0)
		return (1);

	listen (sock, 5);
	while (1) {

		int new_sock;
		socklen_t socklen = sizeof(struct sockaddr_in);
		new_sock = accept(sock, (struct sockaddr *) & client_address, & socklen);
		if (new_sock < 0) {
			if (errno == EINTR)
				continue;
			return (1);
		}
		switch (fork()) {
			case -1 :
				multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to create a new child process");
				return (1);
			case 0 :
				close(sock);
				exit (function(new_sock, & client_address));
			default :
				close(new_sock);
				break;
		}
	}
}






	static int
give_ip_address	(char * hostname, struct in_addr * result)
{
	struct hostent *	host;
	if (hostname == NULL) {
		result -> s_addr = htonl (INADDR_ANY);
		return (0);
	} else {
#ifdef HAVE_INET_ATON
		if (inet_aton (hostname, result) == 0) {
#else
		if ((result -> s_addr = inet_addr (hostname)) == -1) {
#endif
			if ((host = gethostbyname (hostname)) == NULL)
				return (-1);
			result -> s_addr = ((struct in_addr *) (* host -> h_addr_list)) -> s_addr;
		}
	}
	return (0);
}


	static int
give_port_number (char * portname, char * protocol)
{
	struct servent * serv;	
	if ((serv = getservbyname (portname, protocol)) != NULL)
		return (serv -> s_port);
	return (htons (atoi (portname)));
}	


	static void
no_zombie_sigcld_handler (int unused)
{
	while (waitpid(-1, NULL, WNOHANG) >= 0)
		;
}

	static void
empty_sigalrm_handler (int unused)
{
	/* nothing */
}


	static int
change_uid_gid (void)
{
	/* Change if necessary the effective GID and UID */
	if ((multiserv_gid > 0) && (multiserv_gid != getegid())) {
		if (setgid(multiserv_gid) < 0) {
			multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to change Group-ID");
			return (1);
		}
	}
	if ((multiserv_uid > 0) && (multiserv_uid != geteuid())) {
		if (setuid(multiserv_uid) < 0) {
			multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to change User-ID");
			return (1);
		}
	}
	return( 0);	
}


	static int
install_signal_handlers (void)
{
	/* Install SIGCLD handler to avoid zombies */
	if (signal (SIGCLD, no_zombie_sigcld_handler) == SIG_ERR) {
		multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to catch SIGCLD signal");
		return (1);
	}
	/* Install SIGALRM handler for timeout */
	if (signal (SIGALRM, empty_sigalrm_handler) == SIG_ERR) {
		multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to catch SIGALRM signal");
		return (1);
	}
	/* Ignore SIGPIPE, will process EPIPE errors on write */
	if (signal (SIGPIPE, SIG_IGN) == SIG_ERR) {
		multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to ignore SIGPIPE signal");
		return (1);
	}
	return (0);
}


	static int
create_and_bind_socket (int socket_type)
{
	int	sock;
	int	sockopt;
	struct sockaddr_in	server_address;

	if (socket_type == MULTISERV_UDP)
		sock = socket (AF_INET, SOCK_DGRAM, 0);
	else
		sock = socket (AF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to get a socket");
		return (-1);
	}
	
	memset (& server_address, 0, sizeof (server_address));
	server_address.sin_family = AF_INET;
	if (give_ip_address(multiserv_hostname, &(server_address.sin_addr)) < 0) {
		multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unknown server address");
		return (-1);
	}

	if (socket_type == MULTISERV_UDP)
		server_address.sin_port = give_port_number(multiserv_portname, "udp");
	else
		server_address.sin_port = give_port_number(multiserv_portname, "tcp");
	if (server_address.sin_port == 0) {
		multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unknown server port");
		return (-1);
	}
	sockopt = 1;
	setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) & sockopt, sizeof(sockopt));
	
	if (bind(sock, (struct sockaddr *) & server_address, sizeof (struct sockaddr_in)) < 0) {
		multiserv_log(MULTISERV_LOG_ERROR, "multiserv", "unable to use the given server address");
		return (-1);
	}
	return (sock);
}
