/************************************************************************\
**  telnet.c - Telnet Protocol                           			    **
**                                                                      **
**  Copyright (c) 2004 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/

	/* See RFC 854 for information about the TELNET protocol.
	   We implement only two options: ECHO (RFC 857) and
	   SUPPRESS GO AHEAD (RFC 858).
	 */

	#include <errno.h>
	#include <getopt.h>
	#include <signal.h>
	#include <stdarg.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#include <unistd.h>
	#include <arpa/inet.h>
	#include <netinet/in.h>
	#include <sys/socket.h>
	#include <sys/types.h>
	#include <sys/fcntl.h>

	#include "multiserv.h"
	#include "multiserv_utils.h"

	#include "telnet.h"

	#define TELNET_IAC		255
	#define TELNET_DO		253
	#define TELNET_DONT		254
	#define TELNET_WILL		251
	#define TELNET_WONT		252
	#define TELNET_ECHO		1
	#define TELNET_SGA		3
	
#if defined(MULTISERV_TELNET_TCP)

	static char *	login = "/bin/login";
	static int		telnet_dialog (int sock, struct sockaddr_in * unused_client_address);
	static int		get_pseudo_tty (int * master, int * slave);
	static int		send_IAC(int action, int option, int sock);

	static int		recv_and_process_IAC (int sock, char * buffer, int buffer_len);
	static int		process_IAC_and_send (int socket, char * buffer, int buffer_len);


	int
telnet_description (FILE * stream)
{
	return (fprintf(stream, "Telnet Protocol rfc.854 (TCP/IP)"));
}


	int
telnet_main (int argc, char * argv[])
{
	int i;
	
	i = 1;
	while (i < argc) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				case 'l' :
					if ((argv[i][2] != '\0')
					 || (i == argc - 1)) {
						multiserv_log(MULTISERV_LOG_ERROR, "telnet", "Option -l needs an argument");
						return (1);
					 }
					 login = argv[i+1];
					 i += 2;
					continue;
				 case 'h':
				default :
					break;
			}
			multiserv_usage_global_options(argv[0]);
			fprintf(stderr, "  -l <path>  Pathname for access instead of /bin/login\n");
			return (1);
		}
	}
	multiserv_log(MULTISERV_LOG_MESSAGE, "telnet", "TCP server start");
	return (tcp_server(telnet_dialog));
}



	static int
telnet_dialog (int sock, struct sockaddr_in * unused_client_address)
{
	int master_pty;
	int slave_pty;
	pid_t pid;
	/* reception and transmission are relative to the socket*/
	char buffer_rcpt[MULTISERV_BUFFER_LEN];
	char buffer_trns[MULTISERV_BUFFER_LEN];
	int rcpt_end = 0;
	int trns_end = 0;
	fd_set read_set;
	fd_set write_set;
	int	max_fd_num;
	int n;
	
	multiserv_log(MULTISERV_LOG_MESSAGE, "telnet", "New TCP connection");
	if (get_pseudo_tty (& master_pty, & slave_pty) < 0) {
		multiserv_log(MULTISERV_LOG_ERROR, "telnet", "unable to get a pseudo-tty");
		return (1);
	}
	multiserv_debug("telnet", "Got a pseudo TTY");	

	switch (pid = fork()) {
		case -1:	multiserv_log(MULTISERV_LOG_ERROR, "telnet", "unable to start a child process");
					return (1);
		case 0 :	multiserv_debug("telnet", "Child process started")
					close(master_pty);
					dup2(slave_pty, STDIN_FILENO);
					dup2(slave_pty, STDOUT_FILENO);
					dup2(slave_pty, STDERR_FILENO);
					/* Now do not use multiserv_debug("telnet",() stderr is redirected ! */
					setsid();
					tcsetpgrp(0, getpid());
					/* tcsetattr() ? */
					execl(login, login, NULL);
					multiserv_log(MULTISERV_LOG_ERROR, "telnet", "unable to exec login program");
					kill(getppid(), SIGTERM);
					return (1);
		default :	break;
	}
	/* We are in the parent process */
	close (slave_pty);
	
	multiserv_debug("telnet", "Parent process send IACs");
	send_IAC(TELNET_DO,   TELNET_ECHO, sock);
	send_IAC(TELNET_WILL, TELNET_ECHO, sock);
	send_IAC(TELNET_WILL, TELNET_SGA,  sock);

	/* Let's go */

	while (1) {
		FD_ZERO (& read_set);
		FD_ZERO (& write_set);
		max_fd_num = sock > master_pty ? sock : master_pty;
		
		/* any place to receive data from socket? */
		if (rcpt_end < MULTISERV_BUFFER_LEN) {
			FD_SET(sock, & read_set);
		}
		/* any data to send on socket? */
		if (trns_end > 0) {
			FD_SET(sock, & write_set);
		}
		/* any place to read data from terminal */
		if (trns_end < MULTISERV_BUFFER_LEN) {
			FD_SET(master_pty, & read_set);
		}
		/* any data to write on terminal? */
		if (rcpt_end > 0) {
			FD_SET(master_pty, & write_set);
		}
		if (select (max_fd_num+1, & read_set, & write_set, NULL, NULL) <= 0) {
			multiserv_debug("telnet", "select failed");
			break;
		}

		if (FD_ISSET(sock, & read_set)) {
			multiserv_debug("telnet", "Reception from the socket");
			n = recv_and_process_IAC(sock, buffer_rcpt + rcpt_end, MULTISERV_BUFFER_LEN - rcpt_end);
			/* n is the number of bytes added in the buffer */
			if (n < 0)
				break;
			rcpt_end += n; 
		}
		if (FD_ISSET(master_pty, & write_set)) {
			multiserv_debug("telnet", "Writing on the terminal");
			n = write(master_pty, buffer_rcpt, rcpt_end);
			if (n <= 0)
				break;
			rcpt_end -= n;
			if (rcpt_end > 0)
				memmove(&buffer_rcpt[n], buffer_rcpt, rcpt_end);
		}
		
		if (FD_ISSET(master_pty, & read_set)) {
			multiserv_debug("telnet", "Reading from the terminal");
			n = read(master_pty, buffer_trns, MULTISERV_BUFFER_LEN - trns_end);
			if (n <= 0)
				break;
			trns_end += n;
		}
		if (FD_ISSET(sock, & write_set)) {
			multiserv_debug("telnet", "Sending to the socket");
			n = process_IAC_and_send(sock, buffer_trns, trns_end);
			if (n < 0)
				break;
			trns_end -= n;
			if (trns_end > 0)
				memmove(&buffer_trns[n], buffer_trns, trns_end);
		}
	}
	multiserv_debug("telnet","Parent process ends");
	close (master_pty);
	kill(pid, SIGTERM);
	return (0);
}

	static int
get_pseudo_tty (int * master, int * slave)
{
	/* We use the good old "/dev/ptyXX" method, instead of UNIX 98 ptmx,
	  for portability consideration */
	char name[11];
	char letter_list[] = "abcdepqrstuvwxyz";
	char number_list[] = "0123456789abcdef";
	int l, n;
	strcpy (name, "/dev/ptyXX");
	for (l = 0; letter_list[l] != '\0'; l ++) {
		name[8] = letter_list[l];
		for (n = 0; number_list[n] != '\0'; n ++) {
			name[9] = number_list[n];
			* master = open(name, O_RDWR | O_NOCTTY);
			if (* master < 0)
				continue;
	 		name[5] = 't';
			* slave = open(name, O_RDWR);
			if (* slave >= 0)
				return (0);
			close (* master);
			name[5] = 'p';
		}
	}
	return (-1);
}

	static int
send_IAC(int action, int option, int sock)
{
	char buffer[3];
	buffer[0] = TELNET_IAC;
	buffer[1] = action;
	buffer[2] = option;
	return (send(sock, buffer, 3, 0) == 3);
}


	static int
recv_and_process_IAC (int sock, char * buffer, int buffer_len)
{
	int i = 0;
	int j = 0;
	int n;
	static char previous = 0;
	static int iac_found = 0;
	static int remaining = 0;
	static char iac_codes[2];
	
	n = recv(sock, multiserv_buffer, buffer_len, 0);
	if (n <= 0) 
		/* n = 0 -> remote close */
		return (-1);
	
	if (remaining == 1) {
		iac_codes[1] = multiserv_buffer[i++];
		remaining = 0;
		iac_found = 1;
	} else if (remaining == 2) {
		iac_codes[0] = multiserv_buffer[i++];
		remaining --;
		if (i == n)
			return (0);
		iac_codes[1] = multiserv_buffer[i++];
		remaining --;
		iac_found = 1;
	}
	
	while (i < n) { /* n<=buffer_len, i<n and j<=i imply j<buffer_len */
		if (iac_found) {
			iac_found = 0;
			/* process the iac_codes */
			if (iac_codes[0] == (char)TELNET_IAC)
				buffer[j++]= (char)TELNET_IAC;
		}
		if (multiserv_buffer[i] != (char) TELNET_IAC) {
			if (previous == '\r') {
				if ((multiserv_buffer[i] == '\n')
				 || (multiserv_buffer[i] == '\0')) {
					previous = multiserv_buffer[i++];
					continue;
				}
			}
			previous = buffer[j++] = multiserv_buffer[i++]; /* must check c.l.c faq to be sure */
			continue;
		}
		/* TELNET_IAC */
		i ++;
		if (i == n) {
			remaining = 2;
			break;
		}
		iac_codes[0] = multiserv_buffer[i++];
		if (i == n) {
			remaining = 1;
			break;
		}
		iac_codes[0] = multiserv_buffer[i];
		iac_found = 1;
	}
	return (j);
}

	static int
process_IAC_and_send (int socket, char * buffer, int buffer_len)
{
	int i = 0;
	int j = 0;
	int n;
	while (i < buffer_len) {
		if (buffer[i] != (char)TELNET_IAC) {
			multiserv_buffer[j++] = buffer[i++];
		} else {
			if (j > 0) {
				n = send(socket, multiserv_buffer, j, 0);
				if (n <= 0)
					return (-1);
				if (n != j) {
					return (i-j+n);
				}
			}
			if (! send_IAC(TELNET_IAC, TELNET_IAC, socket))
				return(-1);
			j = 0;
			i ++;
		}
	}
	n = 0;
	if (j > 0) {
		n = send(socket, multiserv_buffer, j, 0);
		if (n <= 0)
			return (-1);
	}
	return (i-j+n);
}

#endif
