/************************************************************************\
**  tftp.c - Trivial File Transfert Protocol                            **
**                                                                      **
**  Copyright (c) 2004 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/

	/* See RFC 1350 for information about the TFTP protocol */

	#include <errno.h>
	#include <getopt.h>
	#include <signal.h>
	#include <stdarg.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#include <unistd.h>
	#include <arpa/inet.h>
	#include <netinet/in.h>
	#include <sys/socket.h>
	#include <sys/types.h>
	#include <sys/fcntl.h>

	#include "multiserv.h"
	#include "multiserv_utils.h"

	#include "tftp.h"

	#define BUFFER_LENGTH	4096

#if defined(MULTISERV_TFTP_UDP)
	static int		tftp_udp_dialog			(int socket, struct sockaddr_in * client, int msg_size);
#endif

#if defined(MULTISERV_TFTP_UDP) || defined (MULTISERV_TFTP_TCP)

	static int		tftp_read_request	(int socket, char * filename, int transfert);
	static int		tftp_write_request	(int socket, char * filename, int transfert);
	static int		tftp_put_block		(int socket, char * block, int size, int nb);
	static int		tftp_get_block 		(int socket, char * block, int * block_size, int nb);


	#define TFTP_READ_REQUEST		0x01
	#define TFTP_WRITE_REQUEST		0x02
	#define TFTP_DATA_BLOCK			0x03
	#define TFTP_ACKNOWLEDGE		0x04
	#define TFTP_ERROR				0x05

	#define TFTP_TRANSFERT_ASCII	0x00
	#define TFTP_TRANSFERT_OCTET	0x01



#endif

/***********************************************************************
     TCP/IP routines
 **********************************************************************/
	
#if defined(MULTISERV_TFTP_TCP)

	int
tftp_tcp_main (int argc, char * argv[])
{
	return (1);
}

	int
tftp_tcp_description (FILE * stream)
{
	return (fprintf(stream, "Trivial File Transfert Protocol rfc.1350 (NOT IMPLEMENTED TCP/IP)"));
}

#endif
	
/***********************************************************************
     UDP/IP routines
 **********************************************************************/

#if defined(MULTISERV_TFTP_UDP)

	int
tftp_udp_description (FILE * stream)
{
	return (fprintf(stream, "Trivial File Transfert Protocol rfc.1350 (UDP/IP)"));
}

	static int
tftp_udp_dialog (int socket, struct sockaddr_in * unused_client_address, int msg_size)
{
	int opcode;
	int i;
	char * filename;
	char * mode;
	int transfert;
	
	multiserv_debug("tftp", "starting dialog");
	opcode = ntohs(* (unsigned short int *)(multiserv_buffer));
	if ((opcode != TFTP_READ_REQUEST) && (opcode != TFTP_WRITE_REQUEST)) {
		multiserv_debug("tftp", "wrong opcode");
		return (1);
	}
		
	/* determine the filename and mode */
	i = 2;
	filename = & (multiserv_buffer[i]);
	while ((i < msg_size) && (multiserv_buffer[i] != '\0'))
		i++;
	if (i == msg_size)
		return (1);
	i++;
	mode = & (multiserv_buffer[i]);
	while ((i < msg_size) && (multiserv_buffer[i] != '\0'))
		i++;
	if (i == msg_size)
		return (1);
	if (strcasecmp (mode, "netascii") == 0) {
		transfert = TFTP_TRANSFERT_ASCII;
	} else if (strcasecmp (mode, "octet") == 0) {
		transfert = TFTP_TRANSFERT_OCTET;
	} else {
		multiserv_debug("tftp", "illegal tranfert mode:");
		multiserv_debug("tftp", mode);
		/* The MAIL transfert mode is obsolete */
		*(unsigned short int *) (multiserv_buffer) = htons(TFTP_ERROR);
		*(unsigned short int *) (multiserv_buffer + 2) = htons(4);
		strcpy (multiserv_buffer + 4, "Illegal mode");
		send(socket, multiserv_buffer, 5 + strlen(multiserv_buffer+4), 0);
		close (socket);
		return (1);
	}

	if (opcode == TFTP_READ_REQUEST)
		return (tftp_read_request(socket, filename, transfert));
	return (tftp_write_request(socket, filename, transfert));
}

	int
tftp_udp_main (int argc, char * argv[])
{
	multiserv_log(MULTISERV_LOG_MESSAGE, "tftp", "UDP server start");
	return (udp_server(tftp_udp_dialog));
}  

#endif


/***********************************************************************
     Common routines
 **********************************************************************/


#if defined(MULTISERV_TFTP_UDP) || defined(MULTISERV_TFTP_TCP)

#define BLOCK_LENGTH	512

	static int
tftp_read_request (int socket, char * filename, int transfert)
{
	FILE * fp;
	char block [BLOCK_LENGTH];
	int block_pos;
	int block_num;
	int curr_char;
	int next_char;
	
	multiserv_log(MULTISERV_LOG_MESSAGE, "tftp read", filename);
	if ((fp = fopen (filename, "r")) == NULL) {
		*(unsigned short int *) (multiserv_buffer) = htons(TFTP_ERROR);
		if (errno == EACCES) {
			*(unsigned short int *) (multiserv_buffer + 2) = htons(2);
			strcpy(multiserv_buffer + 4, "access refused");
			multiserv_log(MULTISERV_LOG_MESSAGE, "tftp", "refused");
		} else {
			*(unsigned short int *) (multiserv_buffer + 2) = htons(1);
			strcpy(multiserv_buffer + 4, "file not found");
			multiserv_log(MULTISERV_LOG_MESSAGE, "tftp", "not found");
		}
		send(socket, multiserv_buffer, 5 + strlen(multiserv_buffer+4), 0);
		close(socket);
		return (1);
	}
	
	block_num = 1;
	next_char = -1;
	do {
		block_pos = 0;
		do {
			if (next_char >= 0) {
				block[block_pos++] = next_char;
				next_char = -1;
				continue;
			}
			curr_char = fgetc(fp);
			if (curr_char == EOF) {
				if (ferror(fp)) {
					multiserv_debug("tftp", "  error on file");
					*(unsigned short int *) (multiserv_buffer) = htons(TFTP_ERROR);
					*(unsigned short int *) (multiserv_buffer + 2) = htons(3);
					multiserv_buffer[4] = 0;
					send (socket, multiserv_buffer, 5, 0);
					return (1);
				}
				/* Normal end of file */
				break;
			}
			if (transfert == TFTP_TRANSFERT_ASCII) {
				if (curr_char == '\r') {
					block[block_pos++] = '\r';
					next_char = '\0';
				} else if (curr_char == '\n') {
					block[block_pos++] = '\r';
					next_char = '\n';
				} else {
					block[block_pos++] = curr_char;
				}
			} else {
				block[block_pos++] = curr_char;
			}
		} while (block_pos < BLOCK_LENGTH);
		if (tftp_put_block (socket, block, block_pos, block_num) != 0)
			break;
		block_num ++;
	} while (block_pos == BLOCK_LENGTH);
	multiserv_log(MULTISERV_LOG_MESSAGE, "tftp", "sent");
	fclose (fp);
	close (socket);
	return(0);
}

	static int
tftp_write_request (int socket, char * filename, int transfert)
{
	FILE * fp;
	char block [BLOCK_LENGTH];

	int block_size;
	int block_num;
	int last_char;

	multiserv_log(MULTISERV_LOG_MESSAGE, "tftp write", filename);
	
	fp = fopen (filename, "w");

	if (fp == NULL) {
		multiserv_log(MULTISERV_LOG_MESSAGE, "tftp", "refused");
		*(unsigned short int *) (multiserv_buffer) = htons(TFTP_ERROR);
		*(unsigned short int *) (multiserv_buffer + 2) = htons(2);
		strcpy(multiserv_buffer + 4, "  access refused");
		send(socket, multiserv_buffer, 5 + strlen(multiserv_buffer + 4), 0);
		close(socket);
		return (1);
	}

	block_num = 0;
	last_char = 0;
	do {
		int i;
		if (tftp_get_block(socket, block, & block_size, block_num) != 0)
			break;
		for (i = 0; i < block_size; i ++) {
			if (transfert == TFTP_TRANSFERT_ASCII) {
				if (last_char == '\r') {
					last_char = 0;
					if (block[i] == '\0')
						fputc ('\r', fp);
					else
						fputc ('\n', fp);
					continue;
				}
				if (block[i] == '\r') {
					last_char = '\r';
					continue;
				}
			}
			fputc(block[i], fp);
		}
		block_num ++;
	} while (block_size == BLOCK_LENGTH);
	/* Ack last block */
	*(unsigned short int *) (multiserv_buffer) = htons(TFTP_ACKNOWLEDGE);
	*(unsigned short int *) (multiserv_buffer + 2) = htons(block_num);
	send(socket, multiserv_buffer, 4, 0);
	multiserv_log(MULTISERV_LOG_MESSAGE, "tftp", "received");
	
	close(socket);
	fclose(fp);
	return (0);
}
	
	static int
tftp_put_block (int socket, char * block, int size, int nb)
{
	int i;

	/* We will try 3 retransmission max */
	for (i = 0; i < 3; i ++) {
		/* send a data block */	
		* (unsigned short int *)(multiserv_buffer) = htons(TFTP_DATA_BLOCK);
		* (unsigned short int *)(multiserv_buffer+2) = htons(nb);
		memcpy(multiserv_buffer+4, block, size);
		if (send(socket, multiserv_buffer, size + 4, 0) < 0)
			return (-1);

		/* wait for ack */
		while (1) {
			int result;
			multiserv_debug("tftp", "    waiting for ack");
			alarm(5);
			result = recv(socket, multiserv_buffer, BUFFER_LENGTH, 0);
			alarm(0);
			if (result < 0) {
				if (errno == EINTR)
					/* timeout -> resend */
					break;
				multiserv_debug("tftp", "    abort");
				/* error on recvfrom -> abort */
				return (-1);
			}
			if (ntohs(* (unsigned short int *)(multiserv_buffer))   != TFTP_ACKNOWLEDGE)
				return (-1); /* abort */
			if (ntohs(* (unsigned short int *)(multiserv_buffer+2)) != nb)
			 	/* wrong ack block number */
				break;
			/* Ok ! */
			multiserv_debug("tftp", "    got it");
			return (0);
		}
	}
	return (-1);
}

	static int
tftp_get_block (int socket, char * block, int * block_size, int nb)
{
	int i;

	for (i = 0; i < 3; i ++) {
		
		/* Ack the block n */	
		*(unsigned short int *) (multiserv_buffer) = htons(TFTP_ACKNOWLEDGE);
		*(unsigned short int *) (multiserv_buffer + 2) = htons(nb);
		send(socket, multiserv_buffer, 4, 0);
		
		/* wait for block n+1 */
		while (1) {
				int result;
				multiserv_debug("tftp", "    waiting for block");
				alarm (5);
				result = recv(socket, multiserv_buffer, BUFFER_LENGTH, 0);
				alarm (0);
				if (result < 0) {
					if (errno == EINTR) {
						break; /* timeout, retry */
					}
					multiserv_debug("tftp", "    error, abort");
					return (-1); /* error on recvfrom */
				}
				if (*(unsigned short int *) (multiserv_buffer) != htons(TFTP_DATA_BLOCK))
					return (-1); /* abort */
				if (*(unsigned short int *) (multiserv_buffer+2) != htons(nb+1))
					break; /* one more time */
				memcpy (block, multiserv_buffer + 4, result - 4);
				* block_size = result - 4;
				multiserv_debug("tftp", "    got it");
				return (0);
		}
	}
	return (1);
}

#endif
